// ------------------------
// AUDIO / TRACKLIST LOGIC
// ------------------------
const tracks = [
  { title: "Song Title", src: "audio.mp3" },
];

let current = 0;
let isPlaying = false;
let isRepeating = false;
let isShuffling = false;

const audio = document.getElementById("player");
const trackTitle = document.getElementById("track-title");
const progressBar = document.getElementById("progress-bar");
const tracklistDiv = document.getElementById("tracklist");
const playPauseBtn = document.getElementById("play-pause-btn");

function loadTrack(i, autoplay = false) {
  current = i;
  audio.src = tracks[i].src;
  trackTitle.textContent = tracks[i].title;
  progressBar.value = 0;
  localStorage.setItem("currentTrack", i);

  document.querySelectorAll(".track").forEach((t, index) => 
    t.classList.toggle("active", index === i)
  );

  if (autoplay) {
    audio.play();
    isPlaying = true;
    playPauseBtn.textContent = "⏸";
    localStorage.setItem("musicPlaying", "true");
  }
}

function toggleRepeat() {
  isRepeating = !isRepeating;
  document.getElementById("repeat-btn").classList.toggle("active", isRepeating);
}

function toggleShuffle() {
  isShuffling = !isShuffling;
  document.getElementById("shuffle-btn").classList.toggle("active", isShuffling);
}

function playPause() {
  if (isPlaying) {
    audio.pause();
    isPlaying = false;
    playPauseBtn.textContent = "▶";
    localStorage.setItem("musicPlaying", "false");
  } else {
    audio.play();
    isPlaying = true;
    playPauseBtn.textContent = "⏸";
    localStorage.setItem("musicPlaying", "true");
  }
}

function nextTrack() {
  if (isRepeating) {
    loadTrack(current, true); // replay the current track
  } else if (isShuffling) {
    let next = Math.floor(Math.random() * tracks.length);
    loadTrack(next, true);
  } else {
    loadTrack((current + 1) % tracks.length, true);
  }
}

function prevTrack() {
  loadTrack((current - 1 + tracks.length) % tracks.length, true);
}

audio.addEventListener("ended", () => {
  if (isRepeating) {
    loadTrack(current, true);
  } else {
    nextTrack();
  }
});

audio.addEventListener("timeupdate", () => {
  progressBar.value = (audio.currentTime / audio.duration) * 100 || 0;
  localStorage.setItem("trackTime", audio.currentTime);
});

progressBar.addEventListener("input", () => {
  audio.currentTime = (progressBar.value / 100) * audio.duration;
});

tracks.forEach((track, i) => {
  const button = document.createElement("button");
  button.className = "track";
  button.textContent = track.title;
  button.onclick = () => { 
    loadTrack(i, true); 
  };
  tracklistDiv.appendChild(button);
});

window.addEventListener("load", () => {
  const savedTrack = localStorage.getItem("currentTrack");
  const savedTime = parseFloat(localStorage.getItem("trackTime")) || 0;
  const savedPlaying = localStorage.getItem("musicPlaying") === "true";

  if (savedTrack !== null) loadTrack(Number(savedTrack), savedPlaying);
  else loadTrack(0);

  audio.currentTime = savedTime;

  if (savedPlaying) {
    audio.play();
    isPlaying = true;
    playPauseBtn.textContent = "⏸";
  } else {
    isPlaying = false;
    playPauseBtn.textContent = "▶";
  }
});